/**
 * Lightweight Professional IP Checker Configuration
 * Optimized for performance with global API coverage
 */

// Global Configuration Object
window.IP_CHECKER_CONFIG = {
    // API Keys (keep these for paid services)
    API_KEYS: {
        IPGEOLOCATION: '', // Optional: https://ipgeolocation.io/
        IPSTACK: '', // Optional: https://ipstack.com/
        IPDATA: '' // Optional: https://ipdata.co/
    },

    // Application Settings (optimized)
    SETTINGS: {
        MAX_HISTORY_ITEMS: 15,
        API_TIMEOUT: 5000, // 5 seconds (reduced for speed)
        RETRY_ATTEMPTS: 1,
        CACHE_DURATION: 300000, // 5 minutes (increased for better caching)
        ENABLE_ANALYTICS: false,
        MAP_LAZY_LOAD: true
    },

    // Feature Flags (streamlined)
    FEATURES: {
        EXPORT_FUNCTIONALITY: true,
        PRIVACY_SCORE: true,
        HISTORY_TRACKING: false,
        ADVANCED_ANALYTICS: false,
        SOCIAL_SHARING: false,
        MAP_INTEGRATION: true
    },

    // Privacy Scoring Weights
    PRIVACY_WEIGHTS: {
        PROXY_DETECTED: -30,
        VPN_DETECTED: -40,
        HOSTING_DETECTED: -25,
        CLOUD_PROVIDER: -20,
        MOBILE_CONNECTION: +10,
        ABUSE_CONFIDENCE: -1,
        DATACENTER: -15
    },

    // Working API Endpoints - Global Coverage
    API_ENDPOINTS: {
        // Avoid http-only services in browser (mixed content)
        IPINFO: {
            url: 'https://ipinfo.io/{ip}/json',
            name: 'IPInfo.io',
            timeout: 5000
        },
        IPAPI_CO: {
            url: 'https://ipapi.co/{ip}/json/',
            name: 'IPAPI.co',
            timeout: 5000
        },
        IPWHO_IS: {
            url: 'https://ipwho.is/{ip}',
            name: 'IPWho.is',
            timeout: 5000
        },
        
        // Additional reliable APIs
        IPWHOIS: {
            url: 'https://ipwhois.app/json/{ip}',
            name: 'IP-Whois',
            timeout: 6000
        },
        IP2LOCATION: {
            url: 'https://api.ip2location.io/?ip={ip}&key=demo&format=json',
            name: 'IP2Location',
            timeout: 6000
        },
        GEOJS: {
            url: 'https://get.geojs.io/v1/ip/geo/{ip}.json',
            name: 'GeoJS',
            timeout: 4000
        },
        IPSB: {
            url: 'https://api.ip.sb/geoip/{ip}',
            name: 'IP.sb',
            timeout: 5000
        },
        FREEGEOIP: {
            url: 'https://freegeoip.app/json/{ip}',
            name: 'FreeGeoIP.app',
            timeout: 5000
        },
        DBIP: {
            url: 'https://api.db-ip.com/v2/free/{ip}',
            name: 'DB-IP Free',
            timeout: 5000
        },
        IPLOCATE: {
            url: 'https://www.iplocate.io/api/lookup/{ip}',
            name: 'IPLocate.io',
            timeout: 5000
        },
        GEOLOCATION_DB: {
            url: 'https://geolocation-db.com/json/{ip}',
            name: 'Geolocation-DB',
            timeout: 5000
        },
        SEEIP: {
            url: 'https://ip.seeip.org/geoip/{ip}',
            name: 'seeip.org',
            timeout: 5000
        },
        
        // Optional APIs requiring keys (disabled by default)
        IPDATA: {
            url: 'https://api.ipdata.co/{ip}?api-key=test',
            name: 'IPData.co',
            timeout: 6000
        },
        IPREGISTRY: {
            url: 'https://api.ipregistry.co/{ip}?key=tryout',
            name: 'IPRegistry',
            timeout: 6000
        }
    },

    // IPv6 to IPv4 Mapping Services
    IPV6_SERVICES: {
        TEREDO: {
            pattern: /^2001:0000:/,
            name: 'Teredo Tunneling'
        },
        SIXTO4: {
            pattern: /^2002:/,
            name: '6to4 Tunneling'
        }
    },

    // UI Configuration (minimal)
    UI_CONFIG: {
        ANIMATION_DURATION: 250,
        LOADING_TIMEOUT: 15000, // Reduced timeout
        MAP_DEFAULT_ZOOM: 10,
        MAP_LAZY_THRESHOLD: 1000
    },

    // Streamlined Error Messages
    ERROR_MESSAGES: {
        INVALID_IP: 'Format IP tidak valid',
        NO_DATA: 'Tidak dapat mengambil data IP',
        NETWORK_ERROR: 'Kesalahan jaringan',
        TIMEOUT: 'Request timeout',
        GENERIC: 'Terjadi kesalahan'
    },

    // IP Detection Patterns
    IP_PATTERNS: {
        IPV4: /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/,
        IPV6: /^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$/,
        IPV4_MAPPED: /^::ffff:([0-9]{1,3}\.){3}[0-9]{1,3}$/
    }
};

// Export for Node.js
if (typeof module !== 'undefined' && module.exports) {
    module.exports = window.IP_CHECKER_CONFIG;
}