/**
 * Lightweight Utility Functions for IP Checker
 * Includes IPv6/IPv4 conversion and validation
 */

class IPUtils {
    
    /**
     * Detect IP type and validate
     * @param {string} ip - IP address to validate
     * @returns {object} - Validation result
     */
    static validateIP(ip) {
        if (!ip || typeof ip !== 'string') {
            return { valid: false, type: null, error: 'IP address required' };
        }

        ip = ip.trim();
        const config = window.IP_CHECKER_CONFIG;

        // IPv4 validation
        if (config.IP_PATTERNS.IPV4.test(ip)) {
            return { valid: true, type: 'IPv4', original: ip, error: null };
        }

        // IPv6 validation
        if (config.IP_PATTERNS.IPV6.test(ip)) {
            let subtype = 'standard';
            
            // Check for IPv4-mapped IPv6
            if (config.IP_PATTERNS.IPV4_MAPPED.test(ip)) {
                subtype = 'ipv4-mapped';
            }
            // Check for Teredo
            else if (ip.startsWith('2001:0:') || ip.startsWith('2001::')) {
                subtype = 'teredo';
            }
            // Check for 6to4
            else if (ip.startsWith('2002:')) {
                subtype = '6to4';
            }

            return { valid: true, type: 'IPv6', subtype, original: ip, error: null };
        }

        return { valid: false, type: null, error: 'Invalid IP format' };
    }

    /**
     * Convert IPv6 to IPv4 (if possible)
     * @param {string} ipv6 - IPv6 address
     * @returns {object} - Conversion result
     */
    static ipv6ToIpv4(ipv6) {
        if (!ipv6) return { success: false, ipv4: null, error: 'No IPv6 provided' };

        // If IPv6 contains dotted IPv4 anywhere, extract it
        const dottedMatch = ipv6.match(/(\d{1,3}\.){3}\d{1,3}/);
        if (dottedMatch) {
            const dotted = dottedMatch[0];
            if (this.validateIP(dotted).valid) {
                // Classify mapped vs compatible vs embedded
                const method = ipv6.includes('::ffff:') ? 'IPv4-mapped' : (ipv6.includes('5efe:') ? 'ISATAP (dotted)' : 'Embedded (dotted)');
                return { success: true, ipv4: dotted, method, error: null };
            }
        }

        // IPv4-mapped IPv6 (::ffff:192.168.1.1)
        const ipv4MappedMatch = ipv6.match(/^::ffff:(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})$/);
        if (ipv4MappedMatch) {
            return { 
                success: true, 
                ipv4: ipv4MappedMatch[1], 
                method: 'IPv4-mapped',
                error: null 
            };
        }

        // NAT64 well-known prefix 64:ff9b::/96 (RFC 6052)
        const nat64 = this.extractNAT64IPv4(ipv6);
        if (nat64) {
            return { success: true, ipv4: nat64, method: 'NAT64 (64:ff9b::/96)', error: null };
        }

        // Teredo tunneling (2001:0::/32)
        if (ipv6.toLowerCase().startsWith('2001:0:') || ipv6.toLowerCase().startsWith('2001::')) {
            try {
                const teredoIpv4 = this.extractTeredoIPv4(ipv6);
                if (teredoIpv4) {
                    return { 
                        success: true, 
                        ipv4: teredoIpv4, 
                        method: 'Teredo tunnel',
                        error: null 
                    };
                }
            } catch (e) {
                // Continue to next method
            }
        }

        // 6to4 tunneling (2002::/16)
        if (ipv6.toLowerCase().startsWith('2002:')) {
            try {
                const sixto4IPv4 = this.extract6to4IPv4(ipv6);
                if (sixto4IPv4) {
                    return { 
                        success: true, 
                        ipv4: sixto4IPv4, 
                        method: '6to4 tunnel',
                        error: null 
                    };
                }
            } catch (e) {
                // Continue to next method
            }
        }

        // ISATAP (intra-site automatic tunnel address protocol)
        const isatap = this.extractISATAPIPv4(ipv6);
        if (isatap) {
            return { success: true, ipv4: isatap, method: 'ISATAP', error: null };
        }

        return { 
            success: false, 
            ipv4: null, 
            error: 'Cannot convert this IPv6 to IPv4',
            suggestion: 'This IPv6 address is not IPv4-compatible'
        };
    }

    /**
     * Extract IPv4 from Teredo tunnel
     * @param {string} teredo - Teredo IPv6 address
     * @returns {string|null} - Extracted IPv4
     */
    static extractTeredoIPv4(teredo) {
        // Proper Teredo extraction (RFC 4380):
        // Format: 2001:0000:SSSS:SSSS:Flags:Port:ClientIPv4 (obfuscated)
        // Client IPv4 is bitwise NOT of the last 32 bits
        const hextets = this.expandIPv6ToHextets(teredo);
        if (!hextets) return null;
        if (!(hextets[0] === 0x2001 && hextets[1] === 0x0000)) return null;
        const clientHigh = hextets[6];
        const clientLow = hextets[7];
        let combined = ((clientHigh & 0xffff) << 16) | (clientLow & 0xffff);
        combined = (~combined) >>> 0; // bitwise NOT, keep unsigned 32
        const ip = [
            (combined >>> 24) & 255,
            (combined >>> 16) & 255,
            (combined >>> 8) & 255,
            combined & 255
        ].join('.');
        return this.validateIP(ip).valid ? ip : null;
    }

    /**
     * Extract IPv4 from 6to4 tunnel
     * @param {string} sixto4 - 6to4 IPv6 address
     * @returns {string|null} - Extracted IPv4
     */
    static extract6to4IPv4(sixto4) {
        // 6to4 format: 2002:IPv4::/48
        const match = sixto4.toLowerCase().match(/^2002:([0-9a-f]{4}):([0-9a-f]{4})/);
        if (match) {
            try {
                const hex1 = parseInt(match[1], 16);
                const hex2 = parseInt(match[2], 16);
                
                const ip = [
                    (hex1 >>> 8) & 255,
                    hex1 & 255,
                    (hex2 >>> 8) & 255,
                    hex2 & 255
                ].join('.');
                
                if (this.validateIP(ip).valid) {
                    return ip;
                }
            } catch (e) {
                return null;
            }
        }
        return null;
    }

    /**
     * Extract IPv4 from NAT64 well-known prefix 64:ff9b::/96
     */
    static extractNAT64IPv4(ipv6) {
        const hextets = this.expandIPv6ToHextets(ipv6);
        if (!hextets) return null;
        // Check prefix 64:ff9b::/96 => first two hextets 0x0064, 0xff9b and hextets[2..5] all zeros
        const nat64PrefixOk = (
            hextets[0] === 0x0064 &&
            hextets[1] === 0xff9b &&
            (hextets[2] === 0x0000 || hextets[2] === 0x0001) &&
            hextets[3] === 0 && hextets[4] === 0 && hextets[5] === 0
        );
        if (nat64PrefixOk) {
            const high = hextets[6];
            const low = hextets[7];
            const ip = [
                (high >>> 8) & 255,
                high & 255,
                (low >>> 8) & 255,
                low & 255
            ].join('.');
            return this.validateIP(ip).valid ? ip : null;
        }
        return null;
    }

    /**
     * Extract IPv4 from ISATAP addresses (::5efe:w.x.y.z or ::0:5efe:w.x.y.z)
     */
    static extractISATAPIPv4(ipv6) {
        // First, if dotted IPv4 present, already handled. For hex form, look for hextet 0x5efe at position 6
        const hextets = this.expandIPv6ToHextets(ipv6);
        if (!hextets) return null;
        // Common patterns place 0x5efe in hextet[6]
        if (hextets[6] === 0x5efe) {
            const low = hextets[7];
            // low is 16 bits only, but ISATAP standard embeds dotted decimal; if only hex available, cannot unambiguously reconstruct full IPv4.
            // However, some notations use hex for last two hextets. Try to interpret hextets[5] and hextets[7] as IPv4 when [6]==5efe
            const high = hextets[5];
            const ip = [
                (high >>> 8) & 255,
                high & 255,
                (low >>> 8) & 255,
                low & 255
            ].join('.');
            return this.validateIP(ip).valid ? ip : null;
        }
        return null;
    }

    /**
     * Expand an IPv6 address to 8 hextets (array of numbers)
     */
    static expandIPv6ToHextets(ipv6) {
        if (!ipv6 || typeof ipv6 !== 'string') return null;
        // If contains dotted IPv4, convert dotted to hex hextets
        let v6 = ipv6.toLowerCase();
        const dotted = v6.match(/(\d{1,3}\.){3}\d{1,3}/);
        let tailHextets = [];
        if (dotted) {
            const octets = dotted[0].split('.').map(n => parseInt(n, 10));
            if (octets.some(n => isNaN(n) || n < 0 || n > 255)) return null;
            // Replace dotted part with two hextets
            const h1 = (octets[0] << 8) | octets[1];
            const h2 = (octets[2] << 8) | octets[3];
            v6 = v6.replace(dotted[0], `${h1.toString(16)}:${h2.toString(16)}`);
        }
        // Split on '::'
        const parts = v6.split('::');
        let left = parts[0] ? parts[0].split(':').filter(Boolean) : [];
        let right = parts[1] ? parts[1].split(':').filter(Boolean) : [];
        // Normalize each hextet to number
        const leftNums = left.map(h => parseInt(h, 16));
        const rightNums = right.map(h => parseInt(h, 16));
        if (leftNums.some(isNaN) || rightNums.some(isNaN)) return null;
        const missing = 8 - (leftNums.length + rightNums.length);
        if (missing < 0) return null;
        const zeros = new Array(missing).fill(0);
        const hextets = [...leftNums, ...zeros, ...rightNums];
        if (hextets.length !== 8) return null;
        return hextets.map(n => (n >>> 0) & 0xffff);
    }

    /**
     * Convert IPv4 to IPv6 (IPv4-mapped)
     * @param {string} ipv4 - IPv4 address
     * @returns {string} - IPv6 address
     */
    static ipv4ToIpv6(ipv4) {
        const validation = this.validateIP(ipv4);
        if (!validation.valid || validation.type !== 'IPv4') {
            return null;
        }

        return `::ffff:${ipv4}`;
    }

    /**
     * Format IP information for display
     * @param {string} ip - IP address
     * @returns {object} - Formatted info
     */
    static getIPInfo(ip) {
        const validation = this.validateIP(ip);
        if (!validation.valid) {
            return { error: validation.error };
        }

        const info = {
            address: ip,
            type: validation.type,
            valid: true
        };

        if (validation.type === 'IPv6') {
            info.subtype = validation.subtype;
            
            // Try conversion
            const conversion = this.ipv6ToIpv4(ip);
            if (conversion.success) {
                info.convertedIPv4 = conversion.ipv4;
                info.conversionMethod = conversion.method;
            }
        } else if (validation.type === 'IPv4') {
            info.mappedIPv6 = this.ipv4ToIpv6(ip);
        }

        return info;
    }

    /**
     * Simple privacy score calculator
     * @param {object} data - API response data
     * @returns {number} - Privacy score
     */
    static calculatePrivacyScore(data) {
        let score = 100;
        const weights = window.IP_CHECKER_CONFIG.PRIVACY_WEIGHTS;

        // Check for proxies/VPNs
        if (data.proxy) score += weights.PROXY_DETECTED;
        if (data.hosting) score += weights.HOSTING_DETECTED;
        if (data.mobile) score += weights.MOBILE_CONNECTION;

        // ISP/Organization checks
        const org = (data.org || data.isp || '').toLowerCase();
        if (org.includes('vpn') || org.includes('proxy')) {
            score += weights.VPN_DETECTED;
        }
        if (org.includes('cloud') || org.includes('amazon') || org.includes('google')) {
            score += weights.CLOUD_PROVIDER;
        }
        if (org.includes('datacenter') || org.includes('hosting')) {
            score += weights.DATACENTER;
        }

        return Math.max(0, Math.min(100, score));
    }

    /**
     * Create toast notification
     * @param {string} message - Message to show
     * @param {string} type - Type of toast (success, error, info)
     */
    static showToast(message, type = 'info') {
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        toast.textContent = message;
        
        // Style the toast
        Object.assign(toast.style, {
            position: 'fixed',
            top: '20px',
            right: '20px',
            padding: '12px 20px',
            borderRadius: '8px',
            color: 'var(--text-primary)',
            background: type === 'error' ? 'var(--error-color)' : 
                       type === 'success' ? 'var(--success-color)' : 'var(--accent-green)',
            zIndex: '10000',
            opacity: '0',
            transform: 'translateX(100%)',
            transition: 'all 0.3s ease'
        });

        document.body.appendChild(toast);

        // Animate in
        setTimeout(() => {
            toast.style.opacity = '1';
            toast.style.transform = 'translateX(0)';
        }, 100);

        // Remove after delay
        setTimeout(() => {
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(100%)';
            setTimeout(() => toast.remove(), 300);
        }, 3000);
    }

    /**
     * Debounce function for performance
     * @param {function} func - Function to debounce
     * @param {number} wait - Wait time in ms
     * @returns {function} - Debounced function
     */
    static debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    /**
     * Format coordinates for display
     * @param {number} lat - Latitude
     * @param {number} lng - Longitude
     * @returns {string} - Formatted coordinates
     */
    static formatCoordinates(lat, lng) {
        if (!lat || !lng) return 'N/A';
        return `${parseFloat(lat).toFixed(4)}, ${parseFloat(lng).toFixed(4)}`;
    }

    /**
     * Get user's current IP (simple detection)
     * @returns {Promise<string>} - Current IP
     */
    static async getCurrentIP() {
        try {
            const response = await fetch('https://api.ipify.org?format=json', {
                timeout: 5000
            });
            const data = await response.json();
            return data.ip;
        } catch (error) {
            // Fallback
            try {
                const response = await fetch('https://ipinfo.io/ip');
                return await response.text();
            } catch (e) {
                throw new Error('Cannot detect current IP');
            }
        }
    }

    /**
     * Get client IPv4 and IPv6 (dual-stack detection)
     * @returns {Promise<{ipv4: string|null, ipv6: string|null}>}
     */
    static async getDualStackIPs() {
        const controller4 = new AbortController();
        const to = (ms, ctrl) => setTimeout(() => ctrl.abort(), ms);
        try {
            const p4 = (async () => {
                try {
                    const t4 = to(5000, controller4);
                    const r = await fetch('https://api.ipify.org?format=json', { signal: controller4.signal });
                    clearTimeout(t4);
                    const j = await r.json();
                    return j.ip || null;
                } catch { return null; }
            })();
            const p6 = (async () => {
                try {
                    const ctrl = new AbortController();
                    const t6 = to(5000, ctrl);
                    const r = await fetch('https://api64.ipify.org?format=json', { signal: ctrl.signal });
                    clearTimeout(t6);
                    const j = await r.json();
                    return j.ip || null;
                } catch { return null; }
            })();
            const [ipv4, ipv6] = await Promise.all([p4, p6]);
            return { ipv4, ipv6 };
        } catch {
            return { ipv4: null, ipv6: null };
        }
    }
}

// Export for global use
window.IPUtils = IPUtils;